/**
 * i-net software provides programming examples for illustration only, without warranty
 * either expressed or implied, including, but not limited to, the implied warranties
 * of merchantability and/or fitness for a particular purpose. This programming example
 * assumes that you are familiar with the programming language being demonstrated and
 * the tools used to create and debug procedures. i-net software support professionals
 * can help explain the functionality of a particular procedure, but they will not modify
 * these examples to provide added functionality or construct procedures to meet your
 * specific needs.
 *
 * Copyright © 1999-2026 i-net software GmbH, Berlin, Germany.
**/
package rdc;

import java.net.URL;

import com.inet.report.Area;
import com.inet.report.DatabaseField;
import com.inet.report.Element;
import com.inet.report.Engine;
import com.inet.report.Field;
import com.inet.report.FieldElement;
import com.inet.report.FieldPart;
import com.inet.report.Fields;
import com.inet.report.FormulaField;
import com.inet.report.GeneralProperties;
import com.inet.report.GroupField;
import com.inet.report.Paragraph;
import com.inet.report.PromptField;
import com.inet.report.SQLField;
import com.inet.report.Section;
import com.inet.report.SortField;
import com.inet.report.SpecialField;
import com.inet.report.SummaryField;
import com.inet.report.Text;
import com.inet.report.TextPart;
import com.inet.report.ValueProperties;

/**
 * This example shows how to work with fields in RDC. The most important class for working with fields is
 * com.inet.report.Fields. It works as a container for all the fields that are defined in a report. You can get an
 * instance of it with <code>Field fields = eng.getFields()</code>. In this class, for each kind of field, you can find
 * methods for adding, removing, getting and check off.
 * @see DatabaseField
 * @see FormulaField
 * @see PromptField
 * @see SpecialField
 * @see GroupField
 * @see SummaryField
 * @see SQLField
 * @see SortField
 */
public class WorkingWithFieldsSample extends RDCSample {

    static final String MAINREPORT = "samples/rdc/mainrep.rpt";

    /**
     * Creates a new engine for an existing report and modify it.
     * @param exportFmt the output format (e.g. Engine.EXPORT_PDF)
     * @return the new engine for the report
     */
    @Override
    public Engine createAndFillEngine( String exportFmt ) {
        try {
            //create a new Engine
            Engine eng = new Engine( exportFmt );
            //set the name of the report file
            URL reportURL = getClass().getResource( '/' + MAINREPORT );
            if( reportURL != null ) {
                eng.setReportFile( reportURL );
            } else {
                eng.setReportFile( MAINREPORT );
            }

            //get the fields set
            Fields fields = eng.getFields();

            //Now you can add, get or remove fields.
            //database fields
            //fields.removeDatabaseField(0);
            DatabaseField dbField3 = fields.getDatabaseField( "Customers.Address" );
            DatabaseField dbField5 = fields.getDatabaseField( "Customers.LastYearsSales" );
            fields.getDatabaseField( 0 );
            DatabaseField dbField2 = fields.getDatabaseField( 1 );
            DatabaseField dbField4 = fields.getDatabaseField( "Customers.CustomerID" );
            //prompt field
            PromptField pField = fields.addPromptField( "Selection Formula", "Give CustomerID(s) to show.", Field.NUMBER );
            pField.setAllowMultipleValues( true );
            pField.setPromptValueAsFormula( "[12311,12312,12313,12314,12316,12317,12318,12319,12310,12320]" );//alternatively you can use setPromptValue(Object) in class PromptField
            //formula field
            FormulaField fmField1 =
                fields.addFormulaField( "formula sample 1", "{Customers.CustomerID} in {?Selection Formula}", FormulaField.FORMULA_USER );//For use in selection formula, set formula in crystal syntax only.
            FormulaField fmField2 =
                fields.addFormulaField( "font_color", "if {Customers.CustomerID} mod 2 = 0 then blue else red", FormulaField.FORMULA_PROPERTY );
            //special field
            SpecialField spField1 = fields.getSpecialField( SpecialField.RECORD_NUMBER );
            //summary field
            SummaryField sumField1 = fields.addSummaryField( dbField5, SummaryField.SUM, "running total 1" );
            //sort field
            fields.addSortField( dbField2.getName(), SortField.ASCENDING_ORDER );
            //Group fields are generated automatically, you can get them only.
            //For each Group in report you can get one group field.
            //Try <code>getGroupFieldsCount()</code>
            //and <code>getGroupField(int)</code> both in class <code>Fields</code>

            //display the Customer ID, Record number and address
            Area phArea = eng.getArea( "PH" );
            Section phSec = phArea.getSection( 0 );

            Area dArea = eng.getArea( "D" );
            Section dSec = dArea.getSection( 0 );

            FieldElement fElem1 = dSec.addFieldElement( dbField4, 0, 0, 1000, 500 );
            Text text1 = phSec.addText( 0, 680, 1000, 500 );
            Paragraph para1 = text1.addParagraph();
            TextPart tPart1 = para1.addTextPart( "ID" );
            para1.setHorAlign( GeneralProperties.ALIGN_HORIZONTAL_RIGHT );
            tPart1.setBold( true );
            tPart1.setUnderline( true );
            dSec.addFieldElement( dbField3, 9000, 0, 1500, 500 );
            Text text2 = phSec.addText( 9000, 680, 1000, 500 );
            Paragraph para2 = text2.addParagraph();
            TextPart tPart2 = para2.addTextPart( "Address" );
            tPart2.setBold( true );
            tPart2.setUnderline( true );
            dSec.addFieldElement( spField1, 11000, 0, 500, 500 );
            Text text3 = phSec.addText( 11000, 680, 1000, 500 );
            Paragraph para3 = text3.addParagraph();
            TextPart tPart3 = para3.addTextPart( "Record" );
            tPart3.setBold( true );
            tPart3.setUnderline( true );

            //Display sum
            Area rfArea = eng.getArea( "RF" );
            Section rfSec = rfArea.getSection( 0 );
            rfSec.setSuppress( false );

            Text text4 = rfSec.addText( 2000, 0, 5000, 500 );
            Paragraph para4 = text4.addParagraph();
            para4.addTextPart( "Sum of sales of listed customers: " );
            FieldPart fPart4 = para4.addFieldPart( sumField1 );
            fPart4.setNumberFormatType( ValueProperties.USE_CUSTOM_FORMAT );
            fPart4.setBold( true );
            fPart4.setRounding( 2 );
            fPart4.setNDecimalPlaces( 2 );
            fPart4.setDecimalSeparatorSymbol( "." );
            fPart4.setCurrencySymbolType( ValueProperties.FLOATING_SYMBOL );
            fPart4.setCurrencySymbol( "$" );
            fPart4.setCurrencyPosition( ValueProperties.LEADING_CURRENCY_OUTSIDE_NEGATIVE );

            //use fmField2 to set the fontColor of all FieldElements in detail section
            Element[] elements = dSec.getElements();
            for( int i = 0; i < elements.length; i++ ) {
                if( elements[i] instanceof FieldElement ) {
                    FieldElement fElem = (FieldElement)elements[i];
                    fElem.setFontColorFormula( fmField2 );
                }
            }

            //use prompt and formula field for record selection
            eng.setSF( fmField1.getFormula() );

            return eng;
        } catch( Throwable e ) {
            e.printStackTrace();
            System.exit( 0 );
            return null;
        }
    }

    /**
     * Main method of this sample
     * @param args arguments not used
     */
    public static void main( String[] args ) {
        new WorkingWithFieldsSample().initUI();
    }
}
